#undef MLN_WO_GLOBAL_VARS
#include <string>
#include <libgen.h>
#include <iostream>
#include <fstream>
#include <limits>
#include <mln/core/image/image2d.hh>
#include <mln/io/magick/all.hh>
#include <mln/io/dump/save.hh>
#include <mln/value/rgb8.hh>
#include <scribo/core/def/lbl_type.hh>
#include <scribo/toolchain/content_in_doc.hh>
#include <scribo/toolchain/text_in_doc_preprocess.hh>
#include <mln/convert/to_qimage.hh>
#include <mln/set/inter.hh>
#include <mln/labeling/colorize.hh>

#include <mln/fun/v2v/rgb_to_luma.hh>

#include <scribo/text/look_like_text_lines.hh>

# include <scribo/binarization/kim.hh>
# include <scribo/binarization/niblack.hh>
# include <scribo/binarization/otsu.hh>
# include <scribo/binarization/sauvola.hh>
# include <scribo/binarization/sauvola_ms.hh>
# include <scribo/binarization/wolf.hh>


#include "ocrwin.hh"
#include "lowoverlap.hh"
#include "match.hh"


struct linfo
{
  linfo() : last_taken_id(0) {}

  mln::accu::shape::bbox<point2d> box;
  mln::accu::stat::mean<unsigned> xheight;
  scribo::line_id_t last_taken_id;
};


namespace mln
{

  namespace convert
  {

    template <typename I>
    inline
    QImage to_qimage_(const Image<I>& ima_)
    {
      const I& ima = exact(ima_);
      mln_precondition(ima.is_valid());

      const unsigned
	nrows = geom::nrows(ima),
	ncols = geom::ncols(ima);

      QImage qima(ncols, nrows, QImage::Format_RGB32);
      uchar * ptr_qima = qima.scanLine(0);
      const mln_value(I)* ptr_ima = &ima(ima.domain().pmin());

      unsigned offset = 2 * border::find(ima) + ima.unmorph_().ncols() - ncols;

      for (unsigned row = 0; row < nrows; ++row, ptr_ima += offset)
      {
	for (unsigned col = 0; col < ncols; ++col)
	{
	  const mln::value::rgb8& v = *ptr_ima++;
#  if Q_BYTE_ORDER == Q_LITTLE_ENDIAN
	  // Memory representation : BBGGRRFF
	  *ptr_qima++ = v.blue();
	  *ptr_qima++ = v.green();
	  *ptr_qima = v.red();
	  ptr_qima += 2;
#  else /* Q_BIG_ENDIAN */
	// Memory representation : FFRRGGBB
	  ++ptr_qima;
	  *ptr_qima++ = v.red();
	  *ptr_qima++ = v.green();
	  *ptr_qima = v.blue();
#  endif // ! Q_BYTE_ORDER
	}
      }

      return qima;
    }
  }
}


template <typename L>
struct order_lines_id
{
  order_lines_id(const scribo::line_set<L>& lines)
    : lines_(lines)
  {
  }

  bool operator()(const scribo::line_id_t& l1, const scribo::line_id_t& l2) const
  {
    const unsigned l1_nsites = lines_(l1).bbox().nsites();
    const unsigned l2_nsites = lines_(l2).bbox().nsites();

    if (l1_nsites == l2_nsites)
      return l1 < l2;
    return l1_nsites < l2_nsites;
  }

  scribo::line_set<L> lines_;
};

template <typename L>
std::string find_out_dir(const scribo::line_info<L>& orig_line)
{
  if (orig_line.x_height() <= 30)
    return "small/";
  else if (orig_line.x_height() <= 55)
    return "medium/";
  else //(it->second.xheight > 200)
    return "large/";
}

template <typename L>
std::pair<std::string, std::string>
make_output_names(const std::string& orig_basename,
		  const std::string& reg_basename,
		  const scribo::line_info<L>& orig_line,
		  const std::string& impl)
{
  std::string bdir = impl + "/" + find_out_dir(orig_line);
  std::string orig_name, reg_name;

  // original image.
  {
    std::stringstream str;
    str << orig_basename << ".png" << orig_line.id() << ".png";
    orig_name = str.str();
  }

  // registered image.
  {
    std::stringstream str;
    str << reg_basename << ".png" << orig_line.id() << ".png";
    reg_name = str.str();
  }

  return std::make_pair(orig_name, reg_name);
}

template <typename L>
std::string
make_output_bbox_name(const std::string& basename,
		      const scribo::line_info<L>& orig_line)
{
  // Saving line for original image.
  std::stringstream str;
  str << "bboxes/" << find_out_dir(orig_line);
  str << basename << orig_line.id() << ".txt";
  return str.str();
}


template <typename L>
void save_bboxes(const std::string& basename,
		 const scribo::line_info<L>& orig_line,
		 const mln::box2d& box)
{
  std::ofstream ostr(make_output_bbox_name(basename, orig_line).c_str());
  ostr << box.pmin().row() << " "
       << box.pmin().col() << " "
       << box.pmax().row() << " "
       << box.pmax().col() << " "
       << std::endl;
  ostr.close();
}


template <typename L>
void save_lines(const std::string& orig_basename,
		const std::string& reg_basename,
		const mln::image2d<bool>& orig,
		const mln::image2d<bool>& reg,
		const scribo::line_info<L>& orig_line,
		const box2d& orig_bbox,
		const box2d& reg_bbox,
		const std::string& impl)
{
  using namespace mln;

  // Add border to help OCR.
  box2d orig_bbox_ = orig_bbox;
  orig_bbox_.enlarge(1);
  box2d reg_bbox_ = reg_bbox;
  reg_bbox_.enlarge(1);

  image2d<bool> out_orig(orig_bbox_);
  image2d<bool> out_reg(reg_bbox_);
  data::paste((orig | orig_bbox_), out_orig);
  data::paste((reg | reg_bbox_), out_reg);

  // Saving line for original image.
  std::string bdir = impl + "/" + find_out_dir(orig_line);

  // Orig
  {
    std::stringstream str;
    str << bdir << orig_basename << orig_line.id() << ".png";

    std::cout << "Saving " << str.str() << std::endl;
    mln::io::magick::save(out_orig, str.str().c_str());
  }

  // Reg
  {
    std::stringstream str;
    str << bdir << reg_basename << orig_line.id() << ".png";

    std::cout << "Saving " << str.str() << std::endl;
    mln::io::magick::save(out_reg, str.str().c_str());
  }
}

template <typename L>
void save_ocr(const std::string& basename,
	      const scribo::line_info<L>& orig_line)
{
  using namespace mln;

  std::stringstream str;
  str << "ocr_gt/" << find_out_dir(orig_line);
  str << basename << orig_line.id();

  std::cout << "Saving " << str.str() + ".txt" << std::endl;
  std::fstream out;
  out.open((str.str() + ".txt").c_str(), std::fstream::out | std::fstream::binary | std::fstream::trunc);
  if (!out.is_open())
  {
    std::cout << "Cannot open output file" << std::endl;
    abort();
  }
  out << orig_line.text() << std::endl;
  out.close();
}




template <typename L>
void save_lines(const std::string& orig_basename,
		const std::string& reg_basename,
		const mln::image2d<bool>& orig,
		const mln::image2d<bool>& reg,
		const scribo::line_info<L>& orig_line,
		const linfo& o2r_info,
		const std::string& impl)
{
  save_lines(orig_basename, reg_basename, orig, reg,
	     orig_line, orig_line.bbox(),
	     o2r_info.box.to_result(), impl);
}


int levenshtein(const std::string &s1, const std::string &s2)
{
  std::string::size_type N1 = s1.length();
  std::string::size_type N2 = s2.length();
  std::string::size_type i, j;
  std::vector<int> T(N2+1);

   for ( i = 0; i <= N2; i++ )
      T[i] = i;

   for ( i = 0; i < N1; i++ ) {
      T[0] = i+1;
      int corner = i;
      for ( j = 0; j < N2; j++ ) {
         int upper = T[j+1];
         if ( s1[i] == s2[j] )
            T[j+1] = corner;
         else
	   T[j+1] = std::min(T[j], std::min(upper, corner)) + 1;
         corner = upper;
      }
   }
   return T[N2];
}



template <typename L>
void merge_lines(scribo::line_set<L>& lines,
		 const scribo::line_id_t& l1,
		 const scribo::line_id_t& l2)
{
  scribo::line_info<L>& l1_info = lines(l1);
  scribo::line_info<L>& l2_info = lines(l2);

  if (l2_info.card() > l1_info.card())
  {
    // we transfer data from the largest item to the root one.
    scribo::line_info<L> tmp = l1_info;
    std::swap(l1_info, l2_info);
    l1_info.fast_merge(tmp);

    // We must set manually the tag for lines(l2) since it is
    // not used directly in merge process so its tag cannot be
    // updated automatically.
    l2_info.update_tag(scribo::line::Merged);
    l2_info.set_hidden(true);
  }
  else
    l1_info.fast_merge(l2_info);

  // l1's tag is automatically set to line::Needs_Precise_Stats_Update
  // l2's tag is automatically set to line::Merged
}


void update_billboard(image2d<scribo::def::lbl_type>& reg_billboard,
//		      image2d<mln::value::rgb8>& reg_billboard_debug,
		      const mln::box2d& box, const scribo::line_id_t& id)
{
  mln::draw::box_plain(reg_billboard, box, id.to_equiv());
  // FIXME: too slow?
  // reg_billboard_debug = mln::data::convert(mln::value::rgb8(),
  // 					   mln::data::wrap(mln::value::int_u8(), reg_billboard));
}

template <typename L>
void handle_nooverlap(const mln::image2d<mln::value::rgb8>& input,
		      scribo::line_set<L>& lines,
		      const scribo::line_id_t& l,
		      L& billboard)
{
  std::cout << "Warning  : == no overlap - line " << l << " cannot be attached to "
	    << "a reference line! ==" << std::endl;

  (void) input;
  // mln::box2d domain = lines(l).bbox();
  // domain.enlarge(0, domain.nrows() * 1.5);
  // domain.enlarge(1, 50);
  // domain.crop_wrt(input.domain());

  // // Building debug image.
  // mln::image2d<mln::value::rgb8> tmp(domain);
  // mln::data::paste(input | tmp.domain(), tmp);
  // mln::draw::box(tmp, lines(l).bbox(), mln::literal::green);
  // QImage ima = mln::convert::to_qimage(tmp);

  // // Launching GUI.
  // lowoverlap win("Box not overlapping !", ima);
  // win.exec();

  // Deleting box in billboard.
  std::cout << "Skip line." << std::endl;
  lines(l).set_hidden(true);
  update_billboard(billboard, lines(l).bbox(), 0);
}

template <typename L>
void handle_lowoverlap(const mln::image2d<mln::value::rgb8>& input,
		       const std::set<unsigned>& labels,
		       std::map<scribo::line_id_t, linfo>& bboxes,
		       const scribo::line_set<L>& orig_lines,
		       scribo::line_set<L>& reg_lines,
		       const scribo::line_id_t& l,
		       L& reg_billboard)
{
  struct values
  {
    mln::value::rgb8 v;
    const char *name;
  };
  static const values colors[] = { { mln::literal::blue, "blue" },
				   { mln::literal::red, "red" },
				   { mln::literal::yellow, "yellow" },
				   { mln::literal::orange, "orange" },
				   { mln::literal::pink, "pink" },
				   { mln::literal::cyan, "cyan" },
				   { mln::literal::lime, "lime" },
				   { mln::literal::black, 0 } };

  std::cout << "Warning  : == low overlap - One of the lines detected cannot be attached to "
	    << "a reference line! ==" << std::endl;


  std::vector<std::pair<scribo::line_id_t, const char *> > ids;
  mln::accu::shape::bbox<point2d> domain;

  // Finding debug domain image.
  typedef std::set<unsigned> set_t;
  for (set_t::const_iterator it = labels.begin();
       it != labels.end();
       ++it)
  {
    if (*it == 0)
      continue;
    domain.take(orig_lines(*it).bbox());
  }

  // Listing ids, choosing colors and building debug image.
  box2d b = domain.to_result();
  b.enlarge(0, b.nrows() * 1.5);
  b.enlarge(1, 50);
  b.crop_wrt(input.domain());
  mln::image2d<mln::value::rgb8> tmp(b);
  data::paste(input | tmp.domain(), tmp);
  int i = 0;
  for (set_t::const_iterator it = labels.begin();
       it != labels.end();
       ++it)
  {
    if (!colors[i].name)
      abort();

    if (*it == 0)
      continue;

    ids.push_back(std::make_pair(*it, colors[i].name));
    mln::draw::box(tmp, orig_lines(*it).bbox(), colors[i].v);
    ++i;
  }
  mln::draw::box(tmp, reg_lines(l).bbox(), mln::literal::green);
  QImage ima = mln::convert::to_qimage(tmp);

  // Launching GUI.
  lowoverlap win("Box not overlapping enough", ids, ima);
  win.exec();

  // Accept matching with chosen line.
  if (win.result() == QDialog::Accepted)
  {
    std::cout << "Merging matching with " << win.selected() << std::endl;
    scribo::line_id_t ml = win.selected();
    bboxes[ml].box.take(reg_lines(l).bbox());
    bboxes[ml].xheight.take(reg_lines(l).x_height());

    // Handle line merging
    if (bboxes[ml].last_taken_id != 0)
    {
      // Merge data
      merge_lines(reg_lines, bboxes[ml].last_taken_id, l); // FIXME: line.id() == l ?
      // Update billboard
      update_billboard(reg_billboard, //reg_billboard_debug,
		       bboxes[ml].box.to_result(),
		       bboxes[ml].last_taken_id);
    }
    else
      bboxes[ml].last_taken_id = l;
  }
  else
  {
    std::cout << "Skip line." << std::endl;
    reg_lines(l).set_hidden(true);
    update_billboard(reg_billboard, //reg_billboard_debug,
		     reg_lines(l).bbox(), 0);
  }
}







template <typename L>
void handle_splitlines_overlap(const mln::image2d<mln::value::rgb8>& input,
			       const std::set<unsigned>& labels,
			       std::map<scribo::line_id_t, linfo>& bboxes,
//			       std::map<scribo::line_id_t, scribo::line_id_t>& o2r_ids,
			       scribo::line_set<L>& orig_lines,
			       scribo::line_set<L>& reg_lines,
			       const scribo::line_id_t& l,
			       L& reg_billboard,
			       L& orig_billboard)
{
  (void) orig_lines;
  (void) reg_lines;
  (void) reg_billboard;

  struct values
  {
    mln::value::rgb8 v;
    const char *name;
  };
  static const values colors[] = { { mln::literal::blue, "blue" },
				   { mln::literal::red, "red" },
				   { mln::literal::yellow, "yellow" },
				   { mln::literal::orange, "orange" },
				   { mln::literal::pink, "pink" },
				   { mln::literal::cyan, "cyan" },
				   { mln::literal::lime, "lime" },
				   { mln::literal::black, 0 } };

  std::cout << "Warning  : == split lines overlap - One of the lines detected cannot be attached to "
	    << "a reference line! ==" << std::endl;


  std::vector<std::pair<scribo::line_id_t, const char *> > ids;
  mln::accu::shape::bbox<point2d> domain;

  // Finding debug domain image.
  typedef std::set<unsigned> set_t;
  for (set_t::const_iterator it = labels.begin();
       it != labels.end();
       ++it)
  {
    if (*it == 0)
      continue;
    domain.take(bboxes[*it].box.to_result());
  }

  // Listing ids, choosing colors and building debug image.
  box2d b = domain.to_result();
  b.enlarge(0, b.nrows() * 1.5);
  b.enlarge(1, 50);
  b.crop_wrt(input.domain());
  mln::image2d<mln::value::rgb8> tmp(b);
  data::paste(input | tmp.domain(), tmp);
  int i = 0;
  for (set_t::const_iterator it = labels.begin();
       it != labels.end();
       ++it)
  {
    if (!colors[i].name)
      abort();

    if (*it == 0)
      continue;

    ids.push_back(std::make_pair(*it, colors[i].name));
    mln::draw::box(tmp, bboxes[*it].box.to_result(), colors[i].v);
    ++i;
  }
  mln::draw::box(tmp, orig_lines(l).bbox(), mln::literal::green);
  QImage ima = mln::convert::to_qimage(tmp);

  // Launching GUI.
  lowoverlap win("Split lines overlapping", ids, ima);
  win.exec();

  // Accept matching with chosen line.
  if (win.result() == QDialog::Accepted)
  {
    std::cout << "Merging matching with " << win.selected() << std::endl;
    scribo::line_id_t ml = win.selected();

    // Merge lines like in registered image.
    merge_lines(orig_lines, ml, l);
    update_billboard(orig_billboard,
  		     orig_lines(ml).bbox(),
  		     ml);
  }
  else
  {
    // FAIL
    abort();

    // std::cout << "Skip line." << std::endl;
    // orig_lines(l).set_hidden(true);
    // reg_lines(o2r_bboxes(l).).set_hidden(true);
    // update_billboard(reg_billboard, //reg_billboard_debug,
    // 		     reg_lines(l).bbox(), 0);
  }
}


mln::box2d crop_bbox(const QRect& select, const mln::box2d& box)
{
  qDebug() << "selection = " << select;
  std::cout << "original box = " << box << std::endl;

  mln::box2d res = box;
  res.pmin().row() += select.top();
  res.pmin().col() += select.left();

  res.pmax().row() = select.bottom() + res.pmin().row();
  res.pmax().col() = select.right() + res.pmin().col();

  res.crop_wrt(box);

  return res;
}








match::match(int argc_, char *argv_[], QObject *parent)
  : QObject(parent), argc(argc_), argv(argv_)
{
}





void match::run()
{
  using namespace mln;
  using namespace scribo;

  Magick::InitializeMagick(0);

  std::stringstream str;
  str << basename(argv[2]);
  typedef image2d<scribo::def::lbl_type> L;
  image2d<value::rgb8> reg;
  image2d<value::rgb8> orig;

  QFileInfo reg_info(argv[1]);
  std::string reg_bname = reg_info.fileName().toStdString();
  QFileInfo orig_info(argv[2]);
  std::string orig_bname = orig_info.fileName().toStdString();

  mln::io::magick::load(reg, argv[1]);
  mln::io::magick::load(orig, argv[2]);

  image2d<value::int_u8> reg_gl = data::transform(reg, mln::fun::v2v::rgb_to_luma<value::int_u8>());
  image2d<value::int_u8> orig_gl = data::transform(orig, mln::fun::v2v::rgb_to_luma<value::int_u8>());

  bool denoise = true;
  std::string language = "fra";
  bool find_line_seps = true;
  bool find_whitespace_seps = true;
  bool enable_ocr = false;

  bool enable_deskew = false;
  // Preprocess document


  //====================
  // 1. Line extraction
  //====================

  // Extract lines for registered document
  document<L> reg_doc;
  {
    image2d<bool>
      reg_preproc = toolchain::text_in_doc_preprocess(reg, false, 0,
						      0.34, enable_deskew, false);

    reg_doc = scribo::toolchain::content_in_doc(reg, reg_preproc, denoise,
						find_line_seps, find_whitespace_seps,
						enable_ocr, language, false);
  }

  // Extract lines for original document
  document<L> orig_doc;
  {
    image2d<bool>
      orig_preproc = toolchain::text_in_doc_preprocess(orig, false, 0,
						       0.34, enable_deskew, false);

    orig_doc = scribo::toolchain::content_in_doc(orig, orig_preproc, denoise,
						 find_line_seps, find_whitespace_seps,
						 enable_ocr, language, false);
  }



  //==========================
  // 2. Buildboard creation
  //==========================


  // Registered image billboard
  L reg_billboard(orig.domain());
  std::vector<scribo::line_id_t> reg_ids;
  line_set<L> reg_lines = reg_doc.paragraphs().lines().duplicate();
  {
    data::fill(reg_billboard, 0u);

    order_lines_id<L> func(reg_lines);
    reg_ids.reserve(reg_lines.nelements() + 1);
    for_all_lines(l, reg_lines)
      if (reg_lines(l).is_textline())
	reg_ids.push_back(l);

    // Sort lines by bbox.nelements() and ids.
    std::sort(reg_ids.begin(), reg_ids.end(), func);

    // Draw boxes from the largest to the smallest.
    for (size_t i = 0; i < reg_ids.size(); ++i)
      mln::draw::box_plain(reg_billboard,
			   reg_lines(reg_ids[i]).bbox(), reg_ids[i].to_equiv());

    // Saving billboard.
    mln::io::magick::save(labeling::colorize(value::rgb8(), reg_billboard),
			  str.str() + "_reg_billboard.png");
  }


  // Original image billboard
  L orig_billboard(orig.domain());
  std::vector<scribo::line_id_t> orig_ids;
  line_set<L> orig_lines = orig_doc.paragraphs().lines().duplicate();
  {
    data::fill(orig_billboard, 0u);

    order_lines_id<L> func(orig_lines);
    orig_ids.reserve(orig_lines.nelements() + 1);
    for_all_lines(l, orig_lines)
      if (orig_lines(l).is_textline())
	orig_ids.push_back(l);

    // Sort lines by bbox.nelements() and ids.
    std::sort(orig_ids.begin(), orig_ids.end(), func);

    // Draw boxes from the largest to the smallest.
    for (int i = orig_ids.size() - 1; i >= 0; --i)
      mln::draw::box_plain(orig_billboard,
			   orig_lines(orig_ids[i]).bbox(), orig_ids[i].to_equiv());

    // Saving billboard.
    mln::io::magick::save(labeling::colorize(value::rgb8(), orig_billboard),
			  str.str() + "_orig_billboard.png");
  }



  //======================
  // Processing lines
  //======================

  typedef std::map<scribo::line_id_t, linfo> map_t;
  map_t o2r_bboxes;

  // Processing registered lines from the smallest to the largest. (in
  // order to allow merging with bigger lines).
  for (size_t i = 0; i < reg_ids.size(); ++i)
  {
    line_id_t l = reg_ids[i];

    //==============================================
    // 2. Match registered lines and original lines.
    //==============================================

    // Here, lines broken in several parts in the registered image are
    // merged in order to match the full line in the original image.

    const box2d& b = reg_lines(l).bbox();
    unsigned p[9];
    p[0] = orig_billboard(b.pmin());
    p[1] = orig_billboard.at_(b.pmin().row(), b.pmax().col());
    p[2] = orig_billboard.at_(b.pcenter().row(), b.pcenter().col());
    p[3] = orig_billboard.at_(b.pcenter().row(), b.pmin().col());
    p[4] = orig_billboard.at_(b.pcenter().row(), b.pcenter().col());
    p[5] = orig_billboard.at_(b.pcenter().row(), b.pmax().col());
    p[6] = orig_billboard.at_(b.pmax().row(), b.pmin().col());
    p[7] = orig_billboard.at_(b.pmax().row(), b.pcenter().col());
    p[8] = orig_billboard(b.pmax());

    typedef std::set<unsigned> set_t;
    set_t labels;

    for (unsigned i = 0; i < 9; ++i)
      labels.insert(p[i]);

    // 1 or more labels
    if (labels.size() > 1)
    {
      std::cout << "Processing line " << l << " overlapping several regions...";

      unsigned maj = 0, maj_c = 0;
      for (set_t::const_iterator it = labels.begin();
	   it != labels.end();
	   ++it)
      {
	if (*it == 0)
	  continue;

	box2d inter_ = geom::bbox(mln::set::inter(orig_lines(*it).bbox(), b));
	if (maj_c < inter_.nsites())
	{
	  maj = *it;
	  maj_c = inter_.nsites();
	}
      }
      std::cout << " => Choosing " << maj << " covering " << (maj_c / (float)orig_lines(maj).bbox().nsites()) * 100 << "% of the box" << std::endl;

      // Does it overlap enough ?
      if ((maj_c / (float)orig_lines(maj).bbox().nsites()) < 0.6
	  && (maj_c / (float)b.nsites()) < 0.6)
      {
	handle_lowoverlap(orig, labels, o2r_bboxes, orig_lines, reg_lines, l, reg_billboard);
      }
      else // ok !
      {
	o2r_bboxes[maj].box.take(b);
	o2r_bboxes[maj].xheight.take(reg_lines(l).x_height());

	// Handle line merging
	if (o2r_bboxes[maj].last_taken_id != 0)
	{
	  // Merge data
	  merge_lines(reg_lines, o2r_bboxes[maj].last_taken_id, l);
	  // Update billboard
	  update_billboard(reg_billboard, //reg_billboard_debug,
			   o2r_bboxes[maj].box.to_result(),
			   o2r_bboxes[maj].last_taken_id);
	}
	else
	  o2r_bboxes[maj].last_taken_id = l;
      }

    } // End of if (labels.size()> 1)

      // Background only: no overlap
    else if (labels.size() == 1 && labels.find(0) != labels.end())
    {
      handle_nooverlap(orig, reg_lines, l, reg_billboard);

    } // end of  if (labels.size() == 1 && labels.find(0) != labels.end())

    else // no background but 1 label.
    {
      mln_assertion(labels.size() == 1);
      set_t::const_iterator it = labels.begin();

      o2r_bboxes[*it].box.take(b);
      o2r_bboxes[*it].xheight.take(reg_lines(l).x_height());

      // Handle line merging
      if (o2r_bboxes[*it].last_taken_id != 0)
      {
	// Merge data
	merge_lines(reg_lines, o2r_bboxes[*it].last_taken_id, l);
	// Update billboard
	update_billboard(reg_billboard, //reg_billboard_debug,
			 o2r_bboxes[*it].box.to_result(),
			 o2r_bboxes[*it].last_taken_id);
      }
      else
	o2r_bboxes[*it].last_taken_id = l;

    } // end of case handling.

  }

  // Drawing updated billboard for registered image.
  L vbillboard(orig.domain());
  data::fill(vbillboard, 0);
  for (map_t::const_iterator it = o2r_bboxes.begin();
       it != o2r_bboxes.end(); ++it)
    mln::draw::box_plain(vbillboard, it->second.box.to_result(), it->first.to_equiv());

  // Saving image
  mln::io::magick::save(labeling::colorize(value::rgb8(), vbillboard),
			str.str() + "_reg_billboard_fixed.png");






  std::cout << "======================== LAST STEP ==========================" << std::endl;




  // order to allow merging with bigger lines).
  for (size_t i = 0; i < orig_ids.size(); ++i)
  {
    line_id_t l = orig_ids[i];

    //==============================================
    // 4. Merge split lines in original document
    //==============================================

    // Here, lines broken in several parts in the registered image are
    // merged in order to match the full line in the original image.

    const box2d& b = orig_lines(l).bbox();
    unsigned p[9];
    p[0] = vbillboard(b.pmin());
    p[1] = vbillboard.at_(b.pmin().row(), b.pmax().col());
    p[2] = vbillboard.at_(b.pcenter().row(), b.pcenter().col());
    p[3] = vbillboard.at_(b.pcenter().row(), b.pmin().col());
    p[4] = vbillboard.at_(b.pcenter().row(), b.pcenter().col());
    p[5] = vbillboard.at_(b.pcenter().row(), b.pmax().col());
    p[6] = vbillboard.at_(b.pmax().row(), b.pmin().col());
    p[7] = vbillboard.at_(b.pmax().row(), b.pcenter().col());
    p[8] = vbillboard(b.pmax());

    typedef std::set<unsigned> set_t;
    set_t labels;

    for (unsigned i = 0; i < 9; ++i)
      labels.insert(p[i]);

    // 1 or more labels
    if (labels.size() > 1)
    {
      std::cout << "Processing line " << l << " overlapping several regions...";

      unsigned maj = 0, maj_c = 0;
      for (set_t::const_iterator it = labels.begin();
	   it != labels.end();
	   ++it)
      {
	if (*it == 0)
	  continue;

	box2d inter_ = geom::bbox(mln::set::inter(orig_lines(*it).bbox(), b));
	if (maj_c < inter_.nsites())
	{
	  maj = *it;
	  maj_c = inter_.nsites();
	}
      }
      std::cout << " => Choosing " << maj << " covering " << (maj_c / (float)orig_lines(maj).bbox().nsites()) * 100 << "% of the box" << std::endl;

      // Does it overlap enough ?
      if ((maj_c / (float)orig_lines(maj).bbox().nsites()) < 0.6
	  && (maj_c / (float)b.nsites()) < 0.6)
      {
	if (maj != l)
	  handle_splitlines_overlap(orig, labels, o2r_bboxes, orig_lines, reg_lines, l, reg_billboard, orig_billboard);
      }
      else // ok !
      {
	if (maj != l)
	  merge_lines(orig_lines, maj, l);
      }

    } // End of if (labels.size()> 1)

      // Background only: no overlap
    else if (labels.size() == 1 && labels.find(0) != labels.end())
    {
      handle_nooverlap(orig, orig_lines, l, orig_billboard);
    } // end of  if (labels.size() == 1 && labels.find(0) != labels.end())

    else // no background but 1 label.
    {
      mln_assertion(labels.size() == 1);
      set_t::const_iterator it = labels.begin();

      if (*it != l)
	merge_lines(orig_lines, *it, l);
    } // end of case handling.

  }

  std::cout << "================= OCR ===============" << std::endl;

  scribo::text::look_like_text_lines_inplace(orig_lines);

  // OCR
  text::recognition(orig_lines, "fra");

  std::cout << "================= Binarize ===============" << std::endl;

  image2d<bool>
    sauvola_orig = binarization::sauvola(orig_gl, 51, 0.34),
    sauvolamsk_orig = binarization::sauvola_ms(orig_gl, 51, 3, 0.34, 0.34, 0.34),
    sauvolamskx_orig = binarization::sauvola_ms(orig_gl, 51, 3, 0.2, 0.3, 0.5),
    wolf_orig = binarization::wolf(orig_gl, 51, 0.34),
    otsu_orig = binarization::otsu(orig_gl),
    niblack_orig = binarization::niblack(orig_gl, 51, 0.2),
    kim_orig = binarization::kim(orig_gl, 101, 0.34);

  image2d<bool>
    sauvola_reg = binarization::sauvola(reg_gl, 51, 0.34),
    sauvolamsk_reg = binarization::sauvola_ms(reg_gl, 51, 3, 0.34, 0.34, 0.34),
    sauvolamskx_reg = binarization::sauvola_ms(reg_gl, 51, 3, 0.2, 0.3, 0.5),
    wolf_reg = binarization::wolf(reg_gl, 51, 0.34),
    otsu_reg = binarization::otsu(reg_gl),
    niblack_reg = binarization::niblack(reg_gl, 51, 0.2),
    kim_reg = binarization::kim(reg_gl, 101, 0.34);


  std::cout << "================= Fixing OCR for each line and save lines ===============" << std::endl;

  // for (size_t i = 0; i < orig_ids.size(); ++i)
  // {
  //   line_id_t l = orig_ids[i];

  for_all_lines(l, orig_lines)
  {
    if (!orig_lines(l).is_textline())
      continue;

    // Open Document OCR GT
    std::ifstream ref;
    ref.open(argv[3]);
    char refline[1024];
    std::string newline;

    int d = INT_MAX;
    if (ref.is_open())
    {
      while (!ref.eof())
      {
	ref.getline(refline, 1024);

	int nd = levenshtein(orig_lines(l).text(), refline);
	if (nd < d)
	{
	  newline = refline;
	  d = nd;
	}
      }
    }
    ref.close();

    std::cout << "Matching text " << orig_lines(l).text() << " and " << newline << std::endl;




    if (d != 0)
    {
      // We need to fix OCR output!

      QImage qorig = mln::convert::to_qimage_(orig | orig_lines(l).bbox());
      QImage qreg = mln::convert::to_qimage_(reg | o2r_bboxes[l].box.to_result());
      ocrwin win(qorig, qreg, newline, orig_lines(l).id().to_equiv());
      win.exec();

      if (win.result() == QDialog::Accepted)
      {
	// Accept fixes and save results
	orig_lines(l).update_text(win.text());

	box2d orig_bbox = crop_bbox(win.orig_select(), orig_lines(l).bbox());
	box2d reg_bbox = crop_bbox(win.reg_select(), o2r_bboxes[l].box.to_result());

	std::cout << orig_lines(l).bbox() << " vs " << orig_bbox << " - " << o2r_bboxes[l].box.to_result() << " vs " << reg_bbox << std::endl;

	save_lines(orig_bname, reg_bname, sauvola_orig, sauvola_reg, orig_lines(l), orig_bbox, reg_bbox, "sauvola");
	save_lines(orig_bname, reg_bname, sauvolamsk_orig, sauvolamsk_reg, orig_lines(l), orig_bbox, reg_bbox, "sauvola_msk");
	save_lines(orig_bname, reg_bname, sauvolamskx_orig, sauvolamskx_reg, orig_lines(l), orig_bbox, reg_bbox, "sauvola_mskx");
	save_lines(orig_bname, reg_bname, wolf_orig, wolf_reg, orig_lines(l), orig_bbox, reg_bbox, "wolf");
	save_lines(orig_bname, reg_bname, otsu_orig, otsu_reg, orig_lines(l), orig_bbox, reg_bbox, "otsu");
	save_lines(orig_bname, reg_bname, niblack_orig, niblack_reg, orig_lines(l), orig_bbox, reg_bbox, "niblack");
	save_lines(orig_bname, reg_bname, kim_orig, kim_reg, orig_lines(l), orig_bbox, reg_bbox, "kim");

	// Save OCR ground-truth
	save_ocr(reg_info.baseName().toStdString(), orig_lines(l));

	// Save Bboxes information
	save_bboxes(orig_bname, orig_lines(l), orig_bbox);
	save_bboxes(reg_bname, orig_lines(l), reg_bbox);

	if (win.edit_output())
	{
	  std::cout << "==== Editing lines... === " << std::endl;
	  std::pair<std::string, std::string> res;
	  res = make_output_names(orig_bname, reg_bname, orig_lines(l), "sauvola");
	  std::cout << "Running : " << (std::string("gimp ") + res.first + " " + res.second).c_str() << std::endl;
	  system((std::string("gimp ") + res.first + " " + res.second + "&").c_str());

	  res = make_output_names(orig_bname, reg_bname, orig_lines(l), "sauvola_msk");
	  system((std::string("gimp ") + res.first + " " + res.second + "&").c_str());

	  res = make_output_names(orig_bname, reg_bname, orig_lines(l), "sauvola_mskx");
	  system((std::string("gimp ") + res.first + " " + res.second + "&").c_str());

	  res = make_output_names(orig_bname, reg_bname, orig_lines(l), "wolf");
	  system((std::string("gimp ") + res.first + " " + res.second + "&").c_str());

	  res = make_output_names(orig_bname, reg_bname, orig_lines(l), "otsu");
	  system((std::string("gimp ") + res.first + " " + res.second + "&").c_str());

	  res = make_output_names(orig_bname, reg_bname, orig_lines(l), "niblack");
	  system((std::string("gimp ") + res.first + " " + res.second + "&").c_str());

	  res = make_output_names(orig_bname, reg_bname, orig_lines(l), "kim");
	  system((std::string("gimp ") + res.first + " " + res.second + "&").c_str());
	}
      }
      else
      {
	// OCR gives garbage, ignore this line.
	orig_lines(l).set_hidden(true);
	update_billboard(reg_billboard, o2r_bboxes[l].box.to_result(), 0);
	update_billboard(orig_billboard, orig_lines(l).bbox(), 0);
      }

    }
    else
    {
      // OCR is right, save the lines!
      save_lines(orig_bname, reg_bname, sauvola_orig, sauvola_reg, orig_lines(l), o2r_bboxes[l], "sauvola");
      save_lines(orig_bname, reg_bname, sauvolamsk_orig, sauvolamsk_reg, orig_lines(l), o2r_bboxes[l], "sauvola_msk");
      save_lines(orig_bname, reg_bname, sauvolamskx_orig, sauvolamskx_reg, orig_lines(l), o2r_bboxes[l], "sauvola_mskx");
      save_lines(orig_bname, reg_bname, wolf_orig, wolf_reg, orig_lines(l), o2r_bboxes[l], "wolf");
      save_lines(orig_bname, reg_bname, otsu_orig, otsu_reg, orig_lines(l), o2r_bboxes[l], "otsu");
      save_lines(orig_bname, reg_bname, niblack_orig, niblack_reg, orig_lines(l), o2r_bboxes[l], "niblack");
      save_lines(orig_bname, reg_bname, kim_orig, kim_reg, orig_lines(l), o2r_bboxes[l], "kim");

      // Save OCR ground-truth
      save_ocr(reg_info.baseName().toStdString(), orig_lines(l));

      // Save BBox
      box2d reg_bbox = o2r_bboxes[l].box.to_result();
      save_bboxes(orig_bname, orig_lines(l), orig_lines(l).bbox());
      save_bboxes(reg_bname, orig_lines(l), reg_bbox);

    }
  }



  // Saving billboards.
  mln::io::magick::save(labeling::colorize(value::rgb8(), reg_billboard),
			str.str() + "_reg_billboard_final.png");
  mln::io::magick::save(labeling::colorize(value::rgb8(), orig_billboard),
			str.str() + "_orig_billboard_final.png");


//  mln::io::dump::save(reg_billboard, str.str() + "_reg_billboard_final.dump");

  emit finished();
}

