#include <QtGui>
#include <QtCore>
#include <iostream>
#include <fstream>
#include <mln/core/alias/box2d.hh>
#include <mln/core/image/image2d.hh>
#include <mln/io/magick/all.hh>
#include <mln/data/paste.hh>

#include "extract.hh"

mln::box2d read_bbox(std::ifstream& file)
{
  mln::point2d pmin, pmax;

  // pmin
  file >> pmin.row();
  file >> pmin.col();

  // pmax
  file >> pmax.row();
  file >> pmax.col();

  return mln::box2d(pmin, pmax);
}




extract::extract(int argc_, char *argv_[], QObject *parent)
  : QObject(parent), argc(argc_), argv(argv_),
    bindir_(argv[1]), bboxdir_(argv[2]), outdir_(argv[3])
{

}





void extract::run()
{
  using namespace mln;

  Magick::InitializeMagick(0);

  QDir bdir(bindir_);
  bdir.setFilter(QDir::Files | QDir::Hidden);
  bdir.setSorting(QDir::Size | QDir::Reversed);

  if (!bdir.exists())
  {
    qDebug() << "Invalid bin directory!";
    exit(1);
  }

  // Browsing available binarizations.
  QFileInfoList blist = bdir.entryInfoList();
  for (int i = 0; i < blist.size(); ++i)
  {
    QFileInfo imageInfo = blist.at(i);
    qDebug() << "* Processing " << imageInfo.fileName();
    image2d<bool> bin;
    io::magick::load(bin, imageInfo.absoluteFilePath().toStdString().c_str());

    QStringList subdirs;
    subdirs << "small" << "medium" << "large";

    foreach (QString subdir, subdirs)
    {
      QDir bbdir(bboxdir_ + "/" + subdir);
      if (!bdir.exists())
      {
	qDebug() << "Invalid bboxes directory!";
	exit(1);
      }
      std::cout << "  Looking into " << bbdir.dirName().toStdString() << std::endl;

      bbdir.setFilter(QDir::Files | QDir::Hidden);
      bbdir.setSorting(QDir::Size | QDir::Reversed);
      QStringList filters;
      filters << imageInfo.baseName() + ".*." + imageInfo.completeSuffix().section('.',0,0) + ".txt";
      qDebug() << filters;
      bbdir.setNameFilters(filters);

      QFileInfoList bblist = bbdir.entryInfoList();
      for (int j = 0; j < bblist.size(); ++j)
      {
	QFileInfo bboxInfo = bblist.at(j);


	std::ifstream file(bboxInfo.absoluteFilePath().toStdString().c_str());
	if (! file)
	{
	  std::cerr << "cannot open file '"
		    << bboxInfo.absoluteFilePath().toStdString().c_str() << "'!";
	  abort();
	}

	box2d bbox = read_bbox(file);

	// Make sure There is enough borders around the text to help Tesseract.
	bbox.enlarge(1);

	qDebug() << "  -> Extracting box " << bboxInfo.fileName();

	QString output = outdir_ + "/" + subdir + "/" + bboxInfo.completeBaseName() + ".png";

	image2d<bool> out(bbox);
	data::paste((bin | bbox), out);
	io::magick::save(out, output.toStdString().c_str());

	file.close();
      }
    }
  }

  emit finished();
}
