#include <mln/core/image/image2d.hh>
#include <mln/debug/println.hh>
#include <QtCore>
#include <string>
#include <stdio.h>
#include <vector>
#include <string>
#include <fstream>
#include <iostream>
#include <limits.h>
#include <cassert>

// Colib
#include <narray.h>
#include <nustring.h>

using namespace colib;
using namespace std;

// =============================================================
// Code extracted from Ocropus 0.3.1 (Under Apache License 2.0)
// =============================================================
// Copyright 2006-2008 Deutsches Forschungszentrum fuer Kuenstliche Intelligenz
// or its licensors, as applicable.
//
// You may not use this code except under the terms of the accompanying license.
//
// Licensed under the Apache License, Version 2.0 (the "License"); you
// may not use this code except in compliance with the License. You may
// obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

namespace {
  template<class T, class S> void fill_slice(narray<T> &array,
					     S item,
					     int from_incl,
					     int upto_excl) {
    for (int i = from_incl; i < upto_excl; i++)
      array[i] = item;
  }
}


namespace ocropus
{

    float block_move_edit_cost(nustring &from, nustring &to, float c) {
        floatarray upper, row;
        row.resize(from.length() + 1);
        upper.resize(from.length() + 1);
        for(int j=0;j<=from.length();j++)
	  row[j] = std::min(c, float(j));
        for(int i=1;i<=to.length();i++) {
            swap(row, upper);

            // calculate costs of a sequence of actions
            row[0] = upper[0] + 1;
            for(int j=1;j<=from.length();j++)
	      row[j] = std::min(std::min(upper[j] + 1, upper[j-1] + (to[i-1] == from[j-1] ? 0 : 1)),
				row[j-1] + 1);

            // find the minimum in the current row
            float best = row[0];
            for(int j=1;j<=from.length();j++) {
                if (row[j] < best)
                    best = row[j];
            }

            // now a sequence of actions
            float upper_bound = best + c;

            for(int j=0;j<=from.length();j++) {
                if (row[j] > upper_bound)
                    row[j] = upper_bound;
            }
        }

        //printf_narray("%f", table);
        return row[from.length()];
    }

    /// Same as block_move_edit_cost(), but also records all block movements
    /// (aka cursor jumps) in the form of two integer arrays.
    float block_move_edit_cost_record_jumps(intarray &jumps_from, intarray &jumps_to, nustring &from, nustring &to, float c) {
        floatarray upper(from.length() + 1);
        floatarray row(from.length() + 1);
        narray<intarray> jumps(from.length() + 1);
        for(int j=0;j<=from.length();j++)
	  row[j] = std::min(c, float(j));
        for(int i=1;i<=to.length();i++) {
            swap(row, upper);
            intarray jumps_tmp; // contains what was in jumps[j-1] at previous iteration

            // calculate costs of a sequence of actions
            row[0] = upper[0] + 1;
            for(int j=1;j<=from.length();j++) {
                float go_up_cost = upper[j] + 1;
                float go_diag_cost = upper[j-1] + (to[i-1] == from[j-1] ? 0 : 1);
                float go_left_cost = row[j-1] + 1;

                if (go_up_cost < go_diag_cost && go_up_cost < go_left_cost) {
                    row[j] = go_up_cost;
                    copy(jumps_tmp, jumps[j]);
                } else if (go_left_cost < go_diag_cost) {
                    row[j] = go_left_cost;
                    move(jumps_tmp, jumps[j]);
                    copy(jumps[j], jumps[j-1]);
                } else {
                    row[j] = go_diag_cost;
                    swap(jumps_tmp, jumps[j]);
                }
            }

            // find the minimum in the current row
            float best = row[0];
            int best_index = 0;
            for(int j=1;j<=from.length();j++) {
                if (row[j] < best) {
                    best = row[j];
                    best_index = j;
                }
            }

            // now a sequence of actions
            float upper_bound = best + c;

            for(int j=0;j<=from.length();j++) {
                if (row[j] > upper_bound) {
                    // we jump from best_index to j
                    row[j] = upper_bound;
                    copy(jumps[j], jumps[best_index]);
                    jumps[j].push(best_index);
                    jumps[j].push(j);
                }
            }
        }

        //printf_narray("%f", table);
        intarray &jumplist = jumps[from.length()];
        jumps_from.resize(jumplist.length() / 2);
        makelike(jumps_to, jumps_from);
        for (int i = 0; i < jumps_from.length(); i++) {
            jumps_from[i] = jumplist[2 * i];
            jumps_to[i] = jumplist[2 * i + 1];
        }
        return row[from.length()];
    }


    float edit_cost_for_layout(intarray &jumps_from, intarray &jumps_to, nustring &from, nustring &to, float c) {
        floatarray upper(from.length() + 1);
        floatarray row(from.length() + 1);
        narray<intarray> jumps(from.length() + 1);
        for(int j=0;j<=from.length();j++)
	  row[j] = std::min(c, float(j));
        for(int i=1;i<=to.length();i++) {
            swap(row, upper);
            intarray jumps_tmp; // contains what was in jumps[j-1] at previous iteration

            // calculate costs of a sequence of actions
            row[0] = upper[0] + 1;
            for(int j=1;j<=from.length();j++) {
                float go_up_cost = upper[j] + 1;
                float go_diag_cost = upper[j-1] + (to[i-1] == from[j-1] ? 0 : 1);
                float go_left_cost = row[j-1] + 1;

                if (go_up_cost < go_diag_cost && go_up_cost < go_left_cost) {
                    row[j] = go_up_cost;
                    copy(jumps_tmp, jumps[j]);
                } else if (go_left_cost < go_diag_cost) {
                    row[j] = go_left_cost;
                    move(jumps_tmp, jumps[j]);
                    copy(jumps[j], jumps[j-1]);
                } else {
                    row[j] = go_diag_cost;
                    swap(jumps_tmp, jumps[j]);
                }
            }

            bool can_jump_everywhere = (to[i-1] == nuchar('\n'));
            // find the minimum in the current row
            // (one for all, another ("absolute") for permitted only)
            float best = row[0];
            int best_index = 0;
            float abs_best = row[0];
            int abs_best_index = 0;
            for(int j=1;j<=from.length();j++) {
                if(row[j] < abs_best) {
                    abs_best = row[j];
                    abs_best_index = j;
                }
                if(!can_jump_everywhere && from[j-1] != nuchar('\n'))
                    continue;
                if(row[j] < best) {
                    best = row[j];
                    best_index = j;
                }
            }

            // now a sequence of actions
            float upper_bound = best + c;
            float better_upper_bound = abs_best + c;

            for(int j=0;j<=from.length();j++) {
                if ((j == 0 || from[j-1] != nuchar('\n'))
                && row[j] > better_upper_bound) {
                    // we jump from best_index to j
                    row[j] = better_upper_bound;
                    copy(jumps[j], jumps[abs_best_index]);
                    jumps[j].push(abs_best_index);
                    jumps[j].push(j);
                } else if (row[j] > upper_bound) {
                    // we jump from best_index to j
                    row[j] = upper_bound;
                    copy(jumps[j], jumps[best_index]);
                    jumps[j].push(best_index);
                    jumps[j].push(j);
                }
            }
        }

        //printf_narray("%f", table);
        intarray &jumplist = jumps[from.length()];
        jumps_from.resize(jumplist.length() / 2);
        makelike(jumps_to, jumps_from);
        for (int i = 0; i < jumps_from.length(); i++) {
            jumps_from[i] = jumplist[2 * i];
            jumps_to[i] = jumplist[2 * i + 1];
        }
        return row[from.length()];
    }


    float block_move_edit_distance(nustring &a, nustring &b, float c) {
        return (block_move_edit_cost(a, b, c)
              + block_move_edit_cost(b, a, c)) / 2;
    }

    void analyze_jumps(bytearray &area_covered_by_non_jumps,
                       intarray &from,
                       intarray &to,
                       int source_length) {
        area_covered_by_non_jumps.resize(source_length);
        int njumps = from.length();
//        ASSERT(njumps == to.length());
        if (!njumps) {
            fill(area_covered_by_non_jumps, 1);
            return;
        }
        fill(area_covered_by_non_jumps, 0);
        fill_slice(area_covered_by_non_jumps, 1, 0, from[0]);
        for (int i = 0; i < njumps - 1; i++) {
            fill_slice(area_covered_by_non_jumps, 1, to[i], from[i+1]);
        }
        fill_slice(area_covered_by_non_jumps,1, to[njumps-1], source_length);
    }

    void get_text_jumped_over(nustring &result, bytearray &covered, nustring &text) {
//        CHECK_ARG(covered.length() == text.length());
        result.clear();
        for(int i = 0; i < text.length(); i++) {
            if(!covered[i])
                result.push(text[i]);
        }
    }


  // Compute "roundtrip" edit distance between 2 nustrings.
  // `c' is the cost of block movement (aka cursor jump).
  // \return 4 values:
  //      -  number of jumps (approximates number of layout analysis errors);
  //      -  number of lines present in `a' but not in `b';
  //      -  number of lines present in `b' but not in `a';
  //      -  edit distance between common lines in `a' and `b' without jump costs
  //             (approximates number of OCR errors)
  void edit_cost_with_garbage(nustring& from, nustring& to, const float& c,
			      nustring& nonskipped, nustring& skipped, float& cost, unsigned& editdist)
  {
    intarray jumps_from;
    intarray jumps_to;
    cost = edit_cost_for_layout(jumps_from, jumps_to, from, to, c);

    bytearray covered;
    analyze_jumps(covered, jumps_from, jumps_to, from.length());

    // Reset arguments.
    nonskipped.clear();
    skipped.clear();

    for (int i = 0; i < from.length(); ++i)
      if (covered.at(i) != 0)
	nonskipped.push(from.at(i));
      else
	skipped.push(from.at(i));

    editdist = jumps_from.length();
  }


} // end of namespace ocropus

// ====================================================================
// End of code extracted from Ocropus 0.3.1 (Under Apache License 2.0)
// ====================================================================


// ===============================================
// Code developped by the LRDE (Under GNU GPL v2) based on the
// editdist script in ocropus 0.3.1
// ===============================================
// Copyright (C) 2012 EPITA Research and Development Laboratory (LRDE)
//
// This code is published as free software: you can redistribute it
// and/or modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2 of the
// License.
//
// This code is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License.
// If not, see <http://www.gnu.org/licenses/>.

unsigned
count_newlines(nustring& text)
{
  unsigned result = 0;
  for (int i = 0; i < text.length(); ++i)
    if (text.at(i).ord() == '\n')
      ++result;
  return result;
}


int levenshtein_distance(std::string s, std::string t)
{
  int
    len_s = s.length(),
    len_t = t.length(),
    cost = 0;

  if (len_s == 0)
    return len_t;
  else if (len_t == 0)
    return len_s;

  if (s[0] != t[0])
    cost = 1;

  std::cout << s.erase(0, 1) << " - " << t.erase(0, 1) << std::endl;

  return std::min(std::min(levenshtein_distance(s.erase(0, 1), t) + 1,
			   levenshtein_distance(s, t.erase(0, 1)) + 1),
		  levenshtein_distance(s.erase(0, 1), t.erase(0, 1)) + cost);
}


// int main(int argc, char *argv[])
// {
//   std::fstream in;
//   std::ifstream ref;

//   if (argc < 3)
//   {
//     std::cout << "Usage: " << argv[0] << " <line.txt> <reflines.txt>" << std::endl;
//     return 1;
//   }

//   in.open(argv[1]);
//   ref.open(argv[2]);

//   char output[1024];
//   char refline[1024];

//   //float c = 2;

//   if (in.is_open() && ref.is_open())
//   {
//     in.getline(&output[0], 1024);
//     ref.getline(&refline[0], 1024);

//     // // Do not consider spaces and '\n' as errors so remove them.
//     int j = 0;
//     for (int i = 0; output[i] != '\0'; ++i)
//     {
//       if (output[i] != ' ' && output[i] != '\n')
//       // if ((output[i] >= 'a' && output[i] <= 'z') ||
//       // 	  (output[i] >= 'A' && output[i] <= 'Z') ||
//       // 	  (output[i] >= '0' && output[i] <= '9'))
//     	output[j++] = output[i];
//     }
//     output[j] = '\0';
//     j = 0;
//     for (int i = 0; refline[i] != '\0'; ++i)
//     {
//       if (refline[i] != ' ' && refline[i] != '\n')
//       // if ((refline[i] >= 'a' && refline[i] <= 'z') ||
//       // 	  (refline[i] >= 'A' && refline[i] <= 'Z') ||
//       // 	  (refline[i] >= '0' && refline[i] <= '9'))
//     	refline[j++] = refline[i];
//     }
//     refline[j] = '\0';

//     std::cout << output << " - " << refline << std::endl;

//     // nustring a(output);
//     // nustring b(refline);

//     // unsigned devnull;

//     // nustring a_nonskipped, a_skipped;
//     // float a_cost;

//     // ocropus::edit_cost_with_garbage(a, b, c, a_nonskipped, a_skipped, a_cost, devnull);

//     // nustring b_nonskipped, b_skipped;
//     // float b_cost;
//     // ocropus::edit_cost_with_garbage(b, a, c, b_nonskipped, b_skipped, b_cost, devnull);

//     // nustring a_all, a_bad;
//     // float a_cost2;
//     // unsigned aj;
//     // ocropus::edit_cost_with_garbage(a, b_nonskipped, c, a_all, a_bad, a_cost2, aj);

//     // nustring b_all, b_bad;
//     // float b_cost2;
//     // unsigned bj;
//     // ocropus::edit_cost_with_garbage(b, a_nonskipped, c, b_all, b_bad, b_cost2, bj);

//     // nustring a_all3, a_bad3;
//     // float a_cost3;
//     // unsigned aj3;
//     // ocropus::edit_cost_with_garbage(a_nonskipped, b_nonskipped, c, a_all3, a_bad3, a_cost3, aj3);

//     // nustring b_bad3;
//     // float b_cost3;
//     // unsigned bj3;
//     // ocropus::edit_cost_with_garbage(b_nonskipped, a_nonskipped, c, a_all3, b_bad3, b_cost3, bj3);

//     // std::cout << aj + bj - std::min(aj3, bj3)
//     // 	      << "\t"
//     // 	      << count_newlines(a_skipped)
//     // 	      << "\t"
//     // 	      << count_newlines(b_skipped)
//     // 	      << "\t"
//     // 	      << std::max(a_cost3 - aj3 * c, b_cost3 - bj3 * c)
//     // 	      << std::endl;

//     // std::cout << std::max(a_cost3 - aj3 * c, b_cost3 - bj3 * c)
//     // 	      << std::endl;

//     //std::cout << "Levenshtein: " <<
//     std::cout << levenshtein_distance(output, refline) << std::endl;

//   }
//   ref.close();
//   in.close();

//   return 0;
// }


int levenshtein_distance(QString in, QString ref)
{

  int
    len_in = in.length(),
    len_ref = ref.length();

  // d est un tableau de longueurChaine1+1 rangées et longueurChaine2+1 colonnes
  mln::image2d<unsigned> d(len_in + 1, len_ref + 1);
  mln::data::fill(d, 0);

  for (int i = 1; i <= in.length(); ++i)
    d.at_(i,0) = i;
  for (int i = 1; i <= ref.length(); ++i)
    d.at_(0,i) = i;


  for (int i = 1; i <= in.length(); ++i)
    for (int j = 1; j <= ref.length(); ++j)
    {
      unsigned cost = 1;
      if (in[i - 1] == ref[j - 1])
	cost = 0;

      d.at_(i,j) = std::min(d.at_(i - 1, j    ) + 1,
			    std::min(d.at_(i    , j - 1) + 1,
				     d.at_(i - 1, j - 1) + cost));
    }

  return d.at_(in.length(), ref.length());
}


int main(int argc, char *argv[])
{
  QCoreApplication(argc, argv);

  if (argc < 3)
  {
    std::cout << "Usage: " << argv[0] << " <line.txt> <reflines.txt>" << std::endl;
    return 1;
  }


  QFile fin(argv[1]);
  if (!fin.open(QIODevice::ReadOnly | QIODevice::Text))
  {
    qDebug() << "Cannot open " << argv[1];
    return 1;
  }

  QFile fref(argv[2]);
  if (!fref.open(QIODevice::ReadOnly | QIODevice::Text))
  {
    qDebug() << "Cannot open " << argv[2];
    return 1;
  }


  QTextStream in(&fin);
  in.setCodec("UTF-8");
  QTextStream ref(&fref);
  ref.setCodec("UTF-8");

  QString in_line = in.readLine();
  QString ref_line = ref.readLine();

  // Do not consider multiple spaces as errors so remove them.
  in_line.remove(' ');
  ref_line.remove(' ');

  std::cout << levenshtein_distance(in_line, ref_line) << std::endl;

  fref.close();
  fin.close();

  return 0;
}
